package graph;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Arrays;
import com.mxgraph.model.mxCell;
import com.Ostermiller.util.CSVParser;
import java.io.File;
import java.io.FileInputStream;
import java.util.ArrayList;
import org.apache.commons.io.FileUtils;

/**
 * Dies ist das Herz vom "GraphTester" - der Graph selber, gepeichert als Adjazenzliste.
 * Die Klasse erlaubt durch geeignete Methoden:
 * - die Speicherung als Adjazenzmatrix,
 * - das Hinzufuegen und Loeschen von knoten und Kanten,
 * - das Markieren von Knoten und Kanten,
 * - eine Aussage darueber, ob Knoten oder Kanten enthalten sind und
 * - eine Ausgabe des Graphen in textueller Form sowie als csv-Datei.
 * 
 * Zu Vor- und Nachteilen der Speicherung als Adjazenzliste oder -matrix siehe "LI1_Hintergrund.odt"
 * 
 * @author Dirk Zechnall, Tom Schaller
 * @version 11.07.2019 (v5.0)
 */
public class Graph
{
    private ArrayList<Knoten> kList;
    private ArrayList<Kante> kaList;
    private ArrayList<ArrayList<Kante>> adList; //Adjazenzliste
    private boolean gerichtet;
    private boolean gewichtet;
    private GraphPlotter plotter;

    /**
     * Der Konstruktor erstellt einen neuen Graphen (genauer eine neue Adjazenzliste)
     * @param isGerichtet gibt an, ob es sich um einen gerichteten Graphen handelt
     * @param isGewichtet gibt an, ob die Kanten gewichtet sind.
     */
    public Graph (boolean isGerichtet, boolean isGewichtet) {
        loescheGraph();
        gerichtet = isGerichtet;
        gewichtet = isGewichtet;
    }

    /**
     * Der Konstruktor erstellt einen neuen Graphen (genauer eine neue Adjazenzliste)
     */
    public Graph() {
        loescheGraph();
    }

    /**
     * Löscht alle Knoten und Kanten eines Graphen und stellt auf ungerichtet und ungewichtet zurück.
     */
    public void loescheGraph() {
        gerichtet = false;
        gewichtet = false;
        kList = new ArrayList<Knoten>();
        kaList = new ArrayList<Kante>();
        adList = new ArrayList<ArrayList<Kante>>();
    }

    /**
     * Ein Graph wird aus einem Matrix-Array erstellt.
     *
     * @param   String[][] matrix   Die Matrix
     */
    public boolean erzeugeGraphAusMatrix(String[][] matrix) {
        boolean geschafft = false;
        try {
            int knotenAnzahl = 0;
            geschafft = true;

            // Knoten erstellen
            for(int i = 0; i < matrix.length; i++) {
                if(matrix[i][0].charAt(0) != '#') {
                    addKnoten(new Knoten(Integer.valueOf(matrix[i][0]), Integer.valueOf(matrix[i][1])));
                    knotenAnzahl++;
                }
            }

            // Kanten erstellen
            int k=0;
            for(int i = 0; i < matrix.length; i++) {
                String[] row = matrix[i];
                if(!row[0].isEmpty() && row[0].charAt(0) != '#') { //Ueberpruefung auf Kommentare #
                    for(int j = 2; j < row.length; j++) {
                        if(!(row[j].equals("-"))) {
                            addKante(getKnoten(k), getKnoten(j - 2), Double.valueOf(row[j]));
                        }
                    }
                    k++;
                }
            }
        } catch(Exception e) {
            geschafft=false;
        }
        return geschafft;
    }

    /**
     * Ein Graph wird aus einem Adjazenzlisten-Array erstellt.
     *
     * @param   String[][] liste   Die Liste
     */
    public boolean erzeugeGraphAusAdjazenzliste(String[][] liste) {
        boolean geschafft = false;
        try {
            int knotenAnzahl = 0;
            geschafft = true;
            // Knoten erstellen
            for(int i = 0; i < liste.length; i++) {
                String[] row = liste[i];
                if(row[0].charAt(0) != '#') { // Ueberpruefung auf Kommentare #
                    if(row[0].charAt(0) != '~')
                        addKnoten(new Knoten(Integer.valueOf(row[0]), Integer.valueOf(row[1])));
                }
            }

            int k = 0;
            // Kanten erstellen
            for(int i = 0; i < liste.length; i++) {
                String[] row = liste[i];
                if(row[0].charAt(0) != '#') { // Ueberpruefung auf Kommentare #
                    for(int j = 2; j < row.length; j = j + 2) {
                        addKante(getKnoten(k), getKnoten(Integer.valueOf(row[j])-1), Double.valueOf(row[j + 1]));
                    }
                    k++;
                }
            }
        } catch(Exception e) {
            geschafft=false;
        }
        return geschafft;
    }

    /**
     * Legt fest, ob der Graph gewichtet oder ungewichtet ist.
     *
     * @param  boolean isWeighted
     */
    public void setGewichtet(boolean isGewichtet) {
        gewichtet = isGewichtet;
    }

    /**
     * Gibt das Attribut weighted zurueck.
     *
     * @return  boolean gewichtet (true/false)
     */
    public boolean isGewichtet() {
        return gewichtet;
    }

    // Der Wert directed sollte im nachinein nicht mehr veraendert werden duerfen, da es Auswirkungen auf die Adjazenzliste hat
    // Ist jedoch wahrscheinlich unproblematisch, wenn dieser spaeter geaendert wird, da er sich nur auf neu erzeugte Kanten auswirkt
    /**
     * Legt fest, ob der Graph gerichtet oder ungerichtet ist.
     *
     * @param  boolean isDirected
     */
    public void setGerichtet(boolean isGerichtet) {
        gerichtet = isGerichtet;
    }

    /**
     * Gibt das Attribut directed zurueck.
     *
     * @return  boolean gerichtet (true/false)
     */
    public boolean isGerichtet() {
        return gerichtet;
    }

    // Gerichtet darf im Nachinein nicht veraendert werden, da sonst die Adjazenzliste nicht mehr stimmt! Daher hier keine toggleGerichtet-Methode!!!

    /** Gibt die Nummer eines Knotens zurück
     * @return int Nummer des Knotens (mit 0 beginnend)
     */
    public int getNummer(Knoten k) {
        return kList.indexOf(k);
    }

    /**
     * Die Methode getAdjazenzMatrix() gibt die Adjazenzmatrix zurueck.
     * 
     * @return  double[][] Die AdjazenzMatrix als zweidimensionales Array
     */
    public double[][] getAdjazenzMatrix() {
        double[][] matrix = new double[getAnzahlKnoten()][getAnzahlKnoten()];
        for (int i=0; i<matrix.length; i++){
            for (int j=0; j<matrix[i].length; j++){
                matrix[i][j] = Double.NaN;
            }
        }
        // speichere Liste In Matrix
        for(Kante k : getAlleKanten()) {
            int startKnotenNummer = getNummer(k.getStart());
            int zielKnotenNummer = getNummer(k.getZiel());
            // Wenn keine Kanten existieren, gibt es auch kein Kantengewicht - daher wird hier das Kantengewicht auf NaN ueberprueft
            // man koennte hier auch mit Double-Variablen arbeiten und diese null setzen --> hat jedoch einen deutlich groeßeren Speicheraufwand!
            if(gewichtet) matrix[startKnotenNummer][zielKnotenNummer] = k.getGewicht();
            else matrix[startKnotenNummer][zielKnotenNummer] = 0;

        }
        return matrix;
    }

/**
 * Gibt eine Liste aller Kanten des Graphen zurück.
 * @return Liste aller Kanten
 */
    public ArrayList<Kante> getAlleKanten() {
        return kaList;
    }

    /**
     * Entfernt die Markierung bei allen Knoten des Graphen.
     */
    public void entferneMarkierungBeiAllenKnoten() {
        for (Knoten k : kList) {
            k.setMarkiert(false);
        }
    }

    /**
     * Initialisiert (via init-Methode der Klasse Knoten) alle Knoten des Graphen.
     */
    public void initialisiereAlleKnoten() {
        for (Knoten k : kList) {
            k.init();
        }
    }

    /**
     * Initialisiert (via init-Methode der Klasse Kante) alle Kanten des Graphen.
     */
    public void initialisiereAlleKanten() {
        for (Kante k: kaList) {            
            k.init();
        }
    }

    /**
     * Ueberprueft, ob ein Knoten in der Knotenliste enthalten ist.
     * Sobald in der Knotenliste der Knoten k gefunden wird, wird true ausgegeben.
     * 
     * @param  Knoten k  Der gesuchte Knoten
     * @return  boolean
     */
    public boolean isKnotenEnthalten (Knoten k) {
        return kList.contains(k);
    }


    /**
     * Gibt die Anzahl der Knoten in der knotenliste zurueck
     * 
     * @return  int
     */
    public int getAnzahlKnoten() {
        return kList.size();
    }

    /**
     * Gibt die Knotenliste zurueck. 
     *
     * @return  ArrayList<Knoten>   Die Knotenliste. Falls leer wird null zurueckgegeben
     */
    public ArrayList<Knoten> getAlleKnoten() {
        return kList;
    }

    /**
     * Gibt die Adjazenzliste (Kantenliste) eines Knotens k zurueck, falls k in der Knotenliste vorhanden ist. 
     *
     * @param   Knoten k    Der Knoten, zu dem die Adjazenzliste gesucht wird
     * @return  ArrayList<Listenelement>
     */
    public ArrayList<Knoten> getNachbarKnoten (Knoten k) {
        if(kList.contains(k)) {
            ArrayList<Knoten> nachbarn = new ArrayList<Knoten>();
            for (Kante e : adList.get(kList.indexOf(k))) {
                nachbarn.add(e.getAnderesEnde(k));
            } // end of for
            return nachbarn;
        } else {
            return null;
        }
    }

    /**
     * Gibt die Adjazenzliste (Kantenliste) eines Knotens k zurueck, falls k in der Knotenliste vorhanden ist. 
     *
     * @param   Knoten k    Der Knoten, zu dem die Adjazenzliste gesucht wird
     * @return  ArrayList<Listenelement>
     */
    public ArrayList<Kante> getAusgehendeKanten (Knoten k) {
        if(kList.contains(k)) {
            return adList.get(kList.indexOf(k));
        } else {
            return null;
        }
    }

    /**
     * Gibt die Kantenliste alle Kanten, die von diesem Knoten k ausgehen, zurueck, falls k in der Knotenliste vorhanden ist. 
     *
     * @param   Knoten k    Der Knoten, zu dem die Adjazenzliste gesucht wird
     * @return  ArrayList<Listenelement>
     */
    public ArrayList<Kante> getAusgehendeKanten (int knotennr) {
        if(knotennr < getAnzahlKnoten()) {
            return adList.get(knotennr);
        } else {
            return null;
        }
    }

    /**
     * Gibt die Kantenliste alle Kanten, die zu diesem Knoten k führen, zurück, falls k in der Knotenliste vorhanden ist. 
     *
     * @param   int knotennr    Nummer des Knoten, zu dem die Adjazenzliste gesucht wird
     * @return  ArrayList<Listenelement>
     */
    public ArrayList<Kante> getEingehendeKanten (int knotennr) {
        if(knotennr >= getAnzahlKnoten()) {
            return null;
        }
        return getEingehendeKanten(this.getKnoten(knotennr));
    }

    /**
     * Gibt die Kantenliste alle Kanten, die zu diesem Knoten k führen, zurück, falls k in der Knotenliste vorhanden ist. 
     *
     * @param   Knoten k    Der Knoten, zu dem die Adjazenzliste gesucht wird
     * @return  ArrayList<Listenelement>
     */

    public ArrayList<Kante> getEingehendeKanten (Knoten k) {
        if(!kList.contains(k)) {
            return null;
        }
        if(this.isGerichtet()) {
            ArrayList<Kante> kanten = new ArrayList<Kante>();
            for (Kante e: kaList) {
                if(e.getZiel()== k) {
                    kanten.add(e);
                }
            } // end of for
            return kanten;

        } else {
            return getAusgehendeKanten(k);
        } // end of if-else

    }

    /**
     * Liefert den n. Knoten des Graphen
     * @param int knotennr Nummer der Knoten (beginnend mit 0)
     */
    public Knoten getKnoten(int knotennr) {
        if(knotennr < getAnzahlKnoten()) {
            return kList.get(knotennr);
        } else {
            return null;
        }
    }    

  
    /**
     * Diese Methode erstellt ein Knoten, der dem graphen hinzugefuegt wird.
     * Dabei wird sowohl die Adjazenzliste, als auch die graphische Repraesentation (jgraphx) aktualisiert.
     * 
     * Falls der Knoten schon enthalten ist oder ein Fehler beim hinzufuegen auftritt, wird false zurueckgegeben.
     * 
     *
     * @param   Knoten k    Der Knoten, der hinzugefuegt werden soll
     * @return  boolean     hinzufuegen hat geklappt (true) oder nicht (false)
     */
    public void addKnoten (Knoten k) {
        if (!isKnotenEnthalten(k)){
            kList.add(k);
            adList.add(new ArrayList<Kante>());
        }
    }

    /**
     * Diese Methode entfernt einen Knoten mit dem angegebenen KnotenNamen.
     * Dabei wird nur die Adjazenzliste aktualisiert, die graphische Repraesentation (jgraphx) erst,
     * wenn der Graph neu gezeichnet wird (in der Klasse GraphPlotter).
     * 
     * Falls der Knoten nicht enthalten ist oder ein Fehler beim hinzufuegen auftritt, wird false zurueckgegeben.
     * 
     *
     * @param   String k    Der Knotenname des Knotens, der geloescht werden soll
     * @return  boolean     Entfernen hat geklappt (true) oder nicht (false)
     */
    public void removeKnoten (int knotennr) {
        removeKnoten(getKnoten(knotennr));
    }

    /**
     * Diese Methode entfernt einen Knoten.
     * Dabei wird nur die Adjazenzliste aktualisiert, die graphische Repraesentation (jgraphx) erst,
     * wenn der Graph neu gezeichnet wird (in der Klasse GraphPlotter).
     * 
     * Falls der Knoten nicht enthalten ist oder ein Fehler beim hinzufuegen auftritt, wird false zurueckgegeben.
     *
     * @param   Knoten k    Der zu entfernende Knoten
     * @return  boolean     Entfernen hat geklappt (true) oder nicht (false)
     */
    public boolean removeKnoten (Knoten k) {
        if (!kList.contains(k)) return false;

        int index = kList.indexOf(k);
        for (ArrayList<Kante> a : adList) {
            int index2 = 0;
            while(index2 < a.size()){
                Kante ka = a.get(index2);
                if(ka.getStart()==k || ka.getZiel()==k){
                    if(kaList.contains(ka)) kaList.remove(ka);
                    a.remove(index2);
                } else {
                    index2++;
                }
            }
        }
        adList.remove(index);
        kList.remove(k);

        return true;
    }

    /**
     * Ueberprueft, ob eine Kante im Graphen enthalten ist.
     *
     * @param   Kante e     Die zu suchende Kante
     * @return  boolean     Kante enthalten (true) oder nicht (false)
     */
    public boolean isKanteEnthalten (Kante e) {
        return kaList.contains(e);
    }

    /**
     * Ueberprueft, ob eine Kante mit Start-, Zielnamen und Kantengewicht im Graphen enthalten ist.
     *
     * @param   String startName    Der StartName
     * @param   String zielName     Der StartName
     * @return  boolean     Kante enthalten (true) oder nicht (false)
     */
    public boolean isKanteEnthalten (int startNr, int zielNr) {
        return getKante(startNr, zielNr)!=null;
    }

    /**
     * Ueberprueft, ob eine Kante mit Start- und Zielknoten im Graphen enthalten ist.
     *
     * @param   Knoten start    Der StartKnoten
     * @param   Knoten ziel     Der StartKnoten
     * @return  boolean     Kante enthalten (true) oder nicht (false)
     */
    public boolean isKanteEnthalten (Knoten start, Knoten ziel) {
        return getKante(start, ziel)!=null;
    }

    /**
     * Gibt eine gesuchte Kante aus dem Graphen zurueck.
     *
     * @param   Knoten start    Der StartKnoten
     * @param   Knoten ziel     Der StartKnoten
     * @return  Kante   Die gesuchte Kante
     */
    public Kante getKante (Knoten start, Knoten ziel) {
        
        for(Kante k: adList.get(kList.indexOf(start))) {
            if(k.getStart()==start && k.getZiel() == ziel) {
                return k;
            }
            if(!gerichtet && k.getStart()==ziel && k.getZiel() == start) {
                return k;
            }
        }
        return null;
    }

    /**
     * Gibt eine gesuchte Kante aus dem Graphen zurueck.
     *
     * @param   int startnr    Der Nummer des StartKnoten
     * @param   int zielnr     Die Nummer des Zielknoten
     * @return  Kante   Die gesuchte Kante
     */
    public Kante getKante (int startnr, int zielnr) {
        return getKante(getKnoten(startnr), getKnoten(zielnr));
    }

    /**
     * Fuegt eine Kante dem Graphen hinzu.
     * Dabei wird ueberprueft, ob die Kante schon im Graphen enthalten ist.
     * 
     * Ist der Graph ungerichtet, werden sowohl "Hin-" und "RueckKante" erstellt.
     *
     * @param   Kante e     Die Kante, die hinzugefuegt werden soll
     * @return  boolean   Hinzufuegen hat geklappt (true) oder nicht (false)
     */
    public void addKante(Kante e) {
        adList.get(kList.indexOf(e.getStart())).add(e);
        if(!gerichtet) adList.get(kList.indexOf(e.getZiel())).add(e);
        kaList.add(e);
    }

    /**
     * Fuegt eine Kante dem Graphen hinzu.
     * Dabei wird ueberprueft, ob die Kante schon im Graphen enthalten ist.
     * 
     * Ist der Graph ungerichtet, werden sowohl "Hin-" und "RueckKante" erstellt.
     *
     * @param   Knoten start     Der StartKnoten der Kante, die hinzugefuegt werden soll
     * @param   Knoten ziel     Der ZielKnoten der Kante, die hinzugefuegt werden soll
     * @param   double gewicht     Das Gewicht der Kante, die hinzugefuegt werden soll
     */
    public void addKante(Knoten start, Knoten ziel, double gewicht) {
        if(!kList.contains(start) || !kList.contains(ziel) || getKante(start,ziel) != null) return;
        addKante(new Kante(start,ziel, gewicht));
    }

    /**
     * Entfernt eine Kante aus dem Graphen.
     * Dabei wird ueberprueft, ob die Kante ueberhaupt im Graphen enthalten ist.
     * 
     * Ist der Graph ungerichtet, werden sowohl "Hin-" und "RueckKante" entfernt.
     *
     * @param   Kante e Die zu entfernende Kante
     */
    public void removeKante (Kante e) {
        removeKante(e.getStart(), e.getZiel());

    }

    /**
     * Entfernt eine Kante aus dem Graphen.
     * Dabei wird ueberprueft, ob die Kante ueberhaupt im Graphen enthalten ist.
     * 
     * Ist der Graph ungerichtet, werden sowohl "Hin-" und "RueckKante" entfernt.
     *
     * @param   Knoten start    StartKnotens
     * @param   Knoten ziel     ZielKnotens
     * @return  boolean Entfernen hat geklappt (true) oder nicht (false)
     */
    public void removeKante (Knoten start, Knoten ziel) {
        Kante e1 = getKante(start, ziel);
        Kante e2 = getKante(ziel, start);

        for(ArrayList<Kante> a: adList) {
            a.remove(e1);
            if(!gerichtet) {
                a.remove(e2);
            }
        }
        kaList.remove(e1);
        kaList.remove(e2);
    }

    /**
     * Entfernt eine Kante aus dem Graphen.
     * Dabei wird ueberprueft, ob die Kante ueberhaupt im Graphen enthalten ist.
     * 
     * Ist der Graph ungerichtet, werden sowohl "Hin-" und "RueckKante" entfernt.
     *
     * @param   int start    StartKnotens
     * @param   int ziel     ZielKnotens
     * @return  boolean Entfernen hat geklappt (true) oder nicht (false)
     */
    public void removeKante (int start, int ziel) {
        removeKante(getKnoten(start), getKnoten(ziel));
    }

    /**
     * Ueberprueft, ob die Adjazenzliste leer ist, d.h. keine Knoten im Graphen enthalten sind.
     * 
     * @return  boolean true, wenn die Liste leer ist, sonst false
     */
    public boolean isEmpty() {
        return adList.isEmpty();
    }

    /**
     * Loescht den gesamten Graphen
     */
    public void clear() {
        adList.clear();
        kList.clear();
        kaList.clear();
        // ausgabe(); //Testzwecke
    }

    /**
     * Die Methode erstellt eine CSV-Ausgabe des Graphen entweder als Adjazenzliste oder als Adjazenzmatrix.
     *
     * @param   boolean asMatrix    true, falls die CSV-Ausgabe eine AdjazenzMatrix sein soll, sonst false
     * @return  String  CSV-Ausgabe
     */
    public String toCSVString(boolean asMatrix) {
        String s="";
        if(adList.isEmpty()) return s;
        if(asMatrix){
            double[][] matrix = getAdjazenzMatrix();
            // hier muss ueberprueft werden, ob matrixinhalt ist NaN --> '-' speichern
            for (int i=0; i<matrix.length; i++){
                s+=(kList.get(i).getX())+","+kList.get(i).getY()+",";
                for (int j=0; j<matrix[i].length; j++){
                    if(Double.isNaN(matrix[i][j]))s+="-";
                    else s+=matrix[i][j];
                    if (j!=matrix.length-1) s+=",";
                }
                s+="\n";
            }
        }
        else {
            for(int knr = 0; knr < kList.size(); knr++) {
                Knoten k = kList.get(knr);
                ArrayList<Kante> a = adList.get(knr);
                s+=(k.getX())+","+(k.getY());
                if (a.size()>0) s+=",";
                for (Kante ka : a) {
                    s+="K"+getNummer(ka.getZiel())+","+ka.getGewicht();
                    if (a.get(a.size()-1)!=ka) s+=",";
                }
                s+="\n";
            }
        }
        return s;
    }

    /**
     * Textuelle Repraesentation des Graphen.
     * 
     * @return  String  Der Graph als Stringrepraesentation
     */
    @Override
    public String toString() {
        String s="";
        s+="Adjazenzliste:\n";
        if(adList.isEmpty()) return "Liste ist leer!\n";
        for(int i=0; i< kList.size(); i++) {
            s+="K"+i+kList.get(i).toString()+": ";
            ArrayList<Kante> a =adList.get(i);
            for (Kante k : a) {
                s+=k.toString()+"K"+(kList.indexOf(k.getAnderesEnde(kList.get(i)))+1)+" | ";
            }
            s=s.substring(0,s.length()-3)+"\n";
        }
        return s;
    }


    /**
     * Textuelle Repraesentation des Graphen mit vorangestelltem Text.
     * 
     * @param   String text Der Text, der vorangestellt werden soll
     * @return  String  Der Graph als Stringrepraesentation
     */
    public String toString(String text) {
        return text+"\n"+toString();
    }

    /**
     * Konsolenausgabe der textuellen Repraesentation des Graphen.
     */
    public void ausgabe() {
        System.out.println(toString());
    }

    /**
     * Konsolenausgabe der textuellen Repraesentation des Graphen mit vorangestelltem Text.
     * 
     * @param   String text Der Text, der vorangestellt werden soll
     */
    public void ausgabe(String text) {
        System.out.println(toString(text));
    }
}
