package graph;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;

import com.mxgraph.util.mxConstants;
import com.mxgraph.swing.mxGraphComponent;
import com.mxgraph.model.mxGraphModel;
import com.mxgraph.view.mxGraph;
import com.mxgraph.model.mxCell;
import com.mxgraph.util.mxConstants;
import com.mxgraph.io.mxCodec;
import com.mxgraph.util.mxUtils;
import com.mxgraph.layout.*;
import com.mxgraph.view.mxStylesheet;
import com.Ostermiller.util.CSVParser;
import java.nio.file.*;
import java.net.URI;
import java.util.ArrayList;
import java.util.Map;
import java.util.HashMap;
import java.util.Arrays;
import org.apache.commons.io.FileUtils;

/**
 * Der GraphPlotter ist das Herzstueck der Visualisierung und dient als Schnittstelle zu jgraphx.
 * Hier werden Methoden zur Verfügung gestellt, die:
 * - Knoten und Kanten färben
 * - diese selektieren
 * - Knoten und Kanten erstellen und entfernen (beides sowohl graphisch als auch in der Adjazenzliste)
 * - und die Codierung als csv-Datei übernehmen.
 * 
 * @author Dirk Zechnall
 * @version 11.11.2015 (v5.3)
 */
public class GraphPlotter extends JPanel {
    // Anfang Attribute
    private mxGraph mxgraph;
    private ArrayList<Object> mxknoten;
    private ArrayList<Object> mxkanten;

    private mxGraphComponent graphComponent;
    private mxGraphModel graphModel;
    protected Graph graph;
    private String bildDatei;
    private boolean bildAnzeigen;
    private int vertexSize = 3;
    private boolean showEdgeWeights = false;
    private boolean showVertexValue = false;
    private boolean showVertexText = false;
    private String[] stdFarbenKnoten = {"lightgrey", "#ACFFAB", "#ACFFED", "#ACC5ED", "#ACC582", "#E5C582", "#E5C5C2", "#D19FFD", "#FF9F8E","#FF6158", "#FF61D4", "#FFDE6E", "#770000", "#007700", "#000077", "#777700", "#77a0e0", "#9912E4", "#783410", "#12a94e"};
    private String[] stdFarbenKanten = {"#606060", "#FF5080", "#D0D0D0"};
    private String[] farbenKnoten = stdFarbenKnoten;
    private String[] farbenKanten = stdFarbenKanten;

    private JTextArea jTAMeldungen = new JTextArea("");
    private JScrollPane jTAMeldungenScrollPane = new JScrollPane(jTAMeldungen);

    // Ende Attribute

    /**
     * Der Konstruktor legt sowohl Einstellungen des mxGraphen (Drag&Drop, Editable, ...) als auch des Graphen (gewichtet, gerichtet, ...) fest.
     *
     * @param  boolean isDirected  Gibt an, ob der Graph gerichtet oder ungerichtet ist
     * @param  boolean isWeighted  Gibt an, ob der Graph gewichtet oder ungewichtet ist
     * @param  String hintergrundBild   Gibt den Namen eines Hintergrundbildes an
     */
    public GraphPlotter() {
        super(new BorderLayout(), true);

        bildDatei = "";
        bildAnzeigen = true;
        showVertexValue = false;
        graph    = new Graph();  
        mxgraph  = new mxGraph();
        mxknoten = new ArrayList<Object>();
        mxkanten = new ArrayList<Object>();
        graphComponent = new mxGraphComponent(mxgraph);
        setPreferredSize(new Dimension(1000,1000));
        setOpaque(true);
        setVisible(true);
        setBackground(new Color(0xFFC0C0));
        graphComponent.getViewport().setOpaque(true);
        graphComponent.setBackground(null);
        add(graphComponent, BorderLayout.CENTER);
        
        jTAMeldungen.setText(graph.toString());
        add(jTAMeldungenScrollPane, BorderLayout.SOUTH);



        // Einstellungen des mxGraph
        mxgraph.setCellsEditable(false);
        mxgraph.setAllowDanglingEdges(false);
        mxgraph.setConnectableEdges(false);
        mxgraph.setAllowLoops(false);
        mxgraph.setCellsDeletable(false);
        mxgraph.setCellsCloneable(false);
        mxgraph.setCellsDisconnectable(false);
        //        mxgraph.setCellsResizable(false);
        mxgraph.setAutoSizeCells(false);    
        mxgraph.setDropEnabled(false);
        mxgraph.setSplitEnabled(false);
        mxgraph.setCellsBendable(false);
        mxgraph.setDisconnectOnMove(false);
        // mxgraph.setCellsResizable(false); //ist ev. gut - jedoch sieht man die selektierten Knoten schlechter
        /* #
         * Alles ausprobiert, um das Verbinden von Kanten per Mouse zu verhindern - ohne Erfolg!
         */
        //         graphComponent.setDragEnabled(false);
        //         graphComponent.setConnectable(false);
        //         graphComponent.getConnectionHandler().setEnabled(false);
        // Einstellungen mxGraph zu Ende
        // Standardwerte für Knoten und Kanten
        Map<String, Object> vertexStyle = new HashMap<String, Object>();

        vertexStyle.put(mxConstants.STYLE_SHAPE,   mxConstants.SHAPE_ELLIPSE);
        vertexStyle.put(mxConstants.STYLE_PERIMETER, mxConstants.PERIMETER_ELLIPSE);
        vertexStyle.put(mxConstants.STYLE_STROKECOLOR, "#000080");
        vertexStyle.put(mxConstants.STYLE_STROKEWIDTH, "2");
        vertexStyle.put(mxConstants.STYLE_FONTCOLOR, "#000080");
        vertexStyle.put(mxConstants.STYLE_FILLCOLOR, "lightgrey");

        mxStylesheet styleSheet = mxgraph.getStylesheet();
        styleSheet.setDefaultVertexStyle(vertexStyle);

        Map<String, Object> edgeStyle = new HashMap<String, Object>();

        edgeStyle.put(mxConstants.STYLE_SHAPE,    mxConstants.SHAPE_CONNECTOR);
        edgeStyle.put(mxConstants.STYLE_STROKECOLOR, "#000080");
        edgeStyle.put(mxConstants.STYLE_STROKEWIDTH, "2");
        edgeStyle.put(mxConstants.STYLE_FONTCOLOR, "#000000");
        edgeStyle.put(mxConstants.STYLE_LABEL_BACKGROUNDCOLOR, "#ffffff");
        edgeStyle.put(mxConstants.STYLE_ENDARROW, mxConstants.ARROW_CLASSIC);

        styleSheet = mxgraph.getStylesheet();
        styleSheet.setDefaultEdgeStyle(edgeStyle);  
        
        MouseMotionListener[] mmls = graphComponent.getGraphControl().getMouseMotionListeners();
        for(MouseMotionListener mml:mmls) {
            graphComponent.getGraphControl().removeMouseMotionListener(mml);
        }
       
    }

    public synchronized boolean updateMxGraph() {
        boolean geschafft = false;
        mxgraph.getModel().beginUpdate();
        try {
            // alles löschen
            mxgraph.selectAll();
            Object[] selected = mxgraph.getSelectionCells();
            for(Object o : selected) {
                mxgraph.getModel().remove(o);
            }

            // Knoten erstellen
            ArrayList<Knoten> knoten = graph.getAlleKnoten();
            mxknoten = new ArrayList<Object>();
            int i=1;
            for(Knoten k : knoten) {
                Object k2;
                if (showVertexText) {
                    k2 = mxgraph.insertVertex(mxgraph.getDefaultParent(), null, "K"+i, k.getX(), k.getY(), vertexSize, vertexSize, "");
                } else {
                    if (showVertexValue) {
                        k2 = mxgraph.insertVertex(mxgraph.getDefaultParent(), null, ""+k.getDoubleWert(), k.getX(), k.getY(), vertexSize, vertexSize, "");
                    } 
                    else {
                        k2 = mxgraph.insertVertex(mxgraph.getDefaultParent(), null, null, k.getX(), k.getY(), vertexSize, vertexSize, "");

                    } // end of if-else
                }
                mxknoten.add(k2);
                mxgraph.setCellStyles(mxConstants.STYLE_FILLCOLOR, farbenKnoten[k.getFarbe()], new Object[]{k2}); // Farbe des Knotens aendern
                
                i++;
            }

            // Kanten erstellen
            String s = "verticalAlign=top;verticalLabelPosition=bottom;";
            if(!graph.isGewichtet()) s += "noLabel=1;";
            if(!graph.isGerichtet()) s += "endArrow=0;";

            ArrayList<Kante> kanten = graph.getAlleKanten();
            mxkanten = new ArrayList<Object>();
            for (Kante k: kanten) {
                Object o;
                if(k.getGewicht() == Double.NaN || !showEdgeWeights ) 
                    o = mxgraph.insertEdge(mxgraph.getDefaultParent(), null, null, mxknoten.get(knoten.indexOf(k.getStart())), mxknoten.get(knoten.indexOf(k.getZiel())), s);
                else
                    o = mxgraph.insertEdge(mxgraph.getDefaultParent(), null, k.getGewicht(), mxknoten.get(knoten.indexOf(k.getStart())), mxknoten.get(knoten.indexOf(k.getZiel())), s);

                mxkanten.add(o);
                if (farbenKanten[k.getFarbe()].equals("invisible")) {
                    mxgraph.setCellStyles(mxConstants.STYLE_STROKECOLOR, "black", new Object[]{o}); //changes the color
                    mxgraph.getModel().setVisible(o, false);
                } else {
                    mxgraph.setCellStyles(mxConstants.STYLE_STROKECOLOR, farbenKanten[k.getFarbe()], new Object[]{o}); //changes the color
                }
                mxgraph.setCellStyles(mxConstants.STYLE_STROKEWIDTH, "2", new Object[]{o}); //changes the width
            } // end of for
        }
        catch(Exception e) {
            geschafft = false;
        }
        finally {
            mxgraph.getModel().endUpdate();
            mxParallelEdgeLayout layout = new mxParallelEdgeLayout(mxgraph);
            layout.execute(mxgraph.getDefaultParent());
            this.repaint();
        }
        return geschafft;
    }  

    public synchronized void paint(Graphics g){
        try{
            super.paint(g);
        } catch(Exception e) {
            System.out.println("Fehler: "+e);
        }
    } 
    
    /**
     * Zeige Info in Infozeile an
     * @param text der Infotext
     */
    public void setInfoText(String text) {
        jTAMeldungen.setText(text);
    }

    /**
     * Gibt den Bilddateinamen aus.
     *
     * @return  Path  Name der Bilddatei
     */
    public String getBildDatei() {
        return bildDatei;
    }

    /**
     * Legt den Bilddateinamen fest. Das Bild muss im Ordner images liegen.
     *
     * @param  String dateiName  Der DateiName
     */
    public void setBildDatei(String dateiName) {
        bildDatei = dateiName;
        setBildAnzeigen(this.bildAnzeigen);
    }

    /**
     * Stellt ein, ob das Bild angezeigt wird
     * @param boolean wird das Bild gezeigt?
     */
    public void setBildAnzeigen(boolean sichtbar) {
        this.bildAnzeigen = sichtbar;
        if(!bildDatei.equals("") && bildAnzeigen) {
            ImageIcon img = new ImageIcon("images/"+bildDatei);
            this.graphComponent.setBackgroundImage(img);
            this.setPreferredSize(new Dimension(img.getIconWidth(), img.getIconHeight()));      

        } else {
            this.graphComponent.setBackgroundImage(null);
        } // end of if-else
        this.graphComponent.repaint();
    }

    public void setWerteVonKnotenAnzeigen(boolean sichtbar) {
        this.showVertexValue = sichtbar;
        updateMxGraph();
    }

    // Testzwecke - koennte sonst auch protected sein ...
    /**
     * Gibt den Graphen zurueck.
     * 
     * @return  Graph
     */
    public Graph getGraph() {
        return graph;
    }

    /**
     * Gibt das selektierte Knotenobjekt zurueck.
     * 
     * @return  Object
     */
    public Knoten getSelectedKnoten() {
        if(mxgraph.getModel().isVertex(mxgraph.getSelectionCell()))
            return graph.getKnoten(mxknoten.indexOf(mxgraph.getSelectionCell()));
        else
            return null;
    }

    /**
     * Gibt die selektierte KnotenobjektListe (als Array) zurueck.
     * 
     * @return  Object[]
     */
    public ArrayList<Knoten> getSelectedKnotenListe() {
        Object[] o = mxgraph.getSelectionCells();
        System.out.println(o.length);
        ArrayList<Knoten> k = new ArrayList<Knoten>();
        for (int i = 0; i<o.length ;i++ ) {
            if(mxgraph.getModel().isVertex(o[i]));
            k.add(graph.getKnoten(mxknoten.indexOf(o[i])));
        } // end of for
        return k;
    }

    /**
     * Gibt das selektierte Kantenobjekt zurueck.
     * 
     * @return  Object
     */
    public Kante getSelectedKante() {
        if(mxgraph.getModel().isEdge(mxgraph.getSelectionCell()))
            return graph.getAlleKanten().get((mxkanten.indexOf(mxgraph.getSelectionCell())));
        else
            return null;
    }

    /**
     * Eine CSV-Datei als Liste wird eingelesen (geoeffnet). Der DateiName wird angegeben.
     *
     * @param   String dateiName   Der DateiName
     * @return  boolean geklappt (true/false)
     */ 
    public boolean csvDateiEinlesen(Path dateiName) {
        boolean geschafft = false;
        try {
            CSVParser csvParser = new CSVParser(Files.newInputStream(dateiName));
            if(csvParser != null) {
                String[][] matrix = csvParser.getAllValues();
                csvParser.close();

                graph.loescheGraph();
                farbenKnoten = stdFarbenKnoten;
                farbenKanten = stdFarbenKanten;
                vertexSize = 30;
                graph.setGerichtet(false);
                graph.setGewichtet(false);
                showEdgeWeights = false;
                showVertexText = true;
                showVertexValue = false;
                bildDatei = "";

                geschafft = true;
                int i;
                for(i = 0; i < matrix.length; i++) {
                    String[] row = matrix[i];

                    if(row[0].equals("directed")) {
                        graph.setGerichtet(row[1].charAt(0) == '1');
                    }
                    if(row[0].equals("weighted")) {
                        graph.setGewichtet(row[1].charAt(0) == '1');
                    }
                    if(row[0].equals("showWeights")) {
                        showEdgeWeights = (row[1].charAt(0) == '1');
                    }
                    if(row[0].equals("vertexSize")) {
                        vertexSize = Integer.parseInt(row[1]);
                    }
                    if(row[0].equals("vertexStyle")) {
                        showVertexText = (row[1].charAt(0) == '1');
                        showVertexValue = (row[1].charAt(0) == '2');
                    }
                    if(row[0].equals("image")) {
                        if(row[1].charAt(0) == '0')
                            bildDatei = "";
                        else
                            bildDatei = row[1];
                    }
                    if(row[0].equals("vertexColor")) {
                        farbenKnoten = Arrays.copyOfRange(row, 1, row.length);  
                    }
                    if(row[0].equals("edgeColor")) {
                        farbenKanten = Arrays.copyOfRange(row, 1, row.length);  
                    }
                    if(row[0].equals("matrix") || row[0].equals("list")) {
                        break;
                    }

                }

                if(matrix[i][0].equals("matrix")) {
                    geschafft = graph.erzeugeGraphAusMatrix(Arrays.copyOfRange(matrix,i+1,matrix.length));
                } else {
                    geschafft = graph.erzeugeGraphAusAdjazenzliste(Arrays.copyOfRange(matrix,i+1,matrix.length));
                }


                updateMxGraph();

                // Alle Komponenten entfernen und neu aufbauen
                if(!bildDatei.equals("") && bildAnzeigen) {
                    this.graphComponent.setBackgroundImage(new ImageIcon("images/"+bildDatei));
                } else {
                    this.graphComponent.setBackgroundImage(null);
                } // end of if-else

                repaint();
            }
        }
        catch(Exception e) {
            geschafft = false;
        }
        return geschafft;
    }

    /**
     * Gibt den MXGraphen aus.
     * 
     * @return  mxGraph Der MX-Graph
     */
    public mxGraph getMXGraph() {
        return mxgraph;
    }

    /**
     * Gibt die MXGraph-Component aus.
     * 
     * @return  mxGraphComponent Die MXGraph-Component
     */
    public mxGraphComponent getGraphComponent() {
        return graphComponent;
    }

    /**
     * Setzt die MXGraph-Component.
     *
     * @param  mxGraphComponent graphComponent  Die Komponente
     */
    public void setGraphComponent(mxGraphComponent graphComponent) {
        this.graphComponent = graphComponent;
    }

    /**
     * Setzt die Farben, die für die Colorierung der Knoten und Kanten verwendet werden.
     * @param farben String-Array mit den Farben (kann "red" usw enthalten oder "#FF0000")
     */
    public void setFarben(String[] farben){
        this.farbenKnoten = farben;
    }

    /**
     * Ueberschreibt die Methode toString. Eine String-Repraesentation des GraphPlotters wird ausgegeben.
     *
     * @return  String  Die String-Repraesentation des GraphPlotters
     */
    public String toString() {
        String s = "";
        if(graph.isGerichtet())
            s += "Gerichteter, ";
        else
            s += "Ungerichteter, ";
        if(graph.isGewichtet())
            s += "gewichteter Graph mit Hintergrundbild: ";
        else
            s += "ungewichteter Graph mit Hintergrundbild: ";
        if(bildDatei.equals(""))
            s += " kein Bild!";
        else
            s += bildDatei + "!";
        return s;
    }

    /**
     * Gibt die String-Repraesentation des GraphPlotters auf der Konsole aus.
     */
    public void ausgabe() {
        System.out.println(toString() + "\n"+graph.toString());
    }
    // Ende Methoden
}
