package grapheditor;

import java.awt.*;
import java.awt.event.*;
import java.util.Arrays;
import java.util.ArrayList;
import java.util.Map;
import java.util.HashMap;
import java.awt.BorderLayout;

import javax.swing.*;
import javax.swing.event.*;
import java.io.File;
import java.io.IOException;
import static java.nio.file.StandardCopyOption.*;
import java.nio.file.*;
import java.net.URI;
import java.nio.file.Files;
import java.util.ArrayList;
import org.apache.commons.io.FileUtils;
import com.Ostermiller.util.CSVParser;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.mxgraph.swing.mxGraphComponent;
import com.mxgraph.swing.handler.mxKeyboardHandler;
import com.mxgraph.swing.handler.mxRubberband;
import com.mxgraph.util.mxDomUtils;
import com.mxgraph.util.mxEvent;
import com.mxgraph.util.mxEventObject;
import com.mxgraph.util.mxEventSource.mxIEventListener;
import com.mxgraph.view.mxStylesheet;
import com.mxgraph.view.mxGraph;
import com.mxgraph.view.mxMultiplicity;
import com.mxgraph.util.mxConstants;
import com.mxgraph.model.mxGraphModel;
import com.mxgraph.model.mxCell;
import com.mxgraph.model.mxGeometry;
import com.mxgraph.layout.*;

import graph.*;

/**
 * Die Editor-GUI
 * 
 * @author Dirk Zechnall
 * @version 23.10.2015 (v4.0)
 */
public class EditorGUI extends JFrame {
    // Anfang Attribute
    private mxGraph mxgraph;
    private mxGraphComponent gc;
    private mxMultiplicity[] checks = new mxMultiplicity[1];
    private String guiTitle;
    private String bildDatei="";

    private int nrKnoten=0;
    private boolean weighted = false;
    private boolean directed = false;
    private boolean bildAnzeigen;
    private int vertexSize = 30;
    private boolean showEdgeWeights = false;
    private boolean showVertexValue = false;
    private boolean showVertexText = false;
    private String[] stdFarbenKnoten = {"lightgrey", "#ACFFAB", "#ACFFED", "#ACC5ED", "#ACC582", "#E5C582", "#E5C5C2", "#D19FFD", "#FF9F8E","#FF6158", "#FF61D4", "#FFDE6E", "#770000", "#007700", "#000077", "#777700", "#77a0e0", "#9912E4", "#783410", "#12a94e"};
    private String[] stdFarbenKanten = {"#606060", "#FF5080", "#D0D0D0"};
    private String[] farbenKnoten = stdFarbenKnoten;
    private String[] farbenKanten = stdFarbenKanten;

    private MouseEvent pressed;

    private JPanel      jPGraph = new JPanel(new BorderLayout(), true);
    private JScrollPane jPGraphScrollPane = new JScrollPane(jPGraph);
    private JTextArea   jTAMeldungen = new JTextArea("");
    private JLabel      jLGraphMode = new JLabel();
    private JLabel      jLCopyright = new JLabel();
    private JLabel     jBCopyright2 = new JLabel();
    private JScrollPane jTAMeldungenScrollPane = new JScrollPane(jTAMeldungen);

    private JButton jButtonSpeichereGraph = new JButton();
    private JButton jButtonBackHome = new JButton();
    private JButton jButtonBildLaden = new JButton();
    private JButton jButtonBildLoeschen = new JButton();
    private JButton jButtonDateiEinlesen = new JButton();
    private JButton jButtonNeuerGraph = new JButton();

    private JFileChooser jFileChooser1 = new JFileChooser();
    private JFileChooser jFileChooser2 = new JFileChooser();
    private JFileChooser jFileChooser3 = new JFileChooser();

    private JMenuItem jPMKanteJMenuItemSetzeKantengewicht = new JMenuItem("Setze Kantengewicht");
    private JMenuItem jPMKanteJMenuItemLöscheausgewählteObjekte = new JMenuItem("Lösche ausgewählte Objekte");
    // Ende Attribute

    public EditorGUI() { 
        // Frame-Initialisierung
        super("Graph-Editor" +" - "+"ungespeichertes Dokument");
        guiTitle = "Graph-Editor";
        int frameWidth = 688; 
        int frameHeight = 522;
        setSize(frameWidth, frameHeight);
        Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
        int x = (d.width - getSize().width) / 2;
        int y = (d.height - getSize().height) / 2;
        setLocation(x, y);
        setResizable(true);
        Container cp = getContentPane();
        cp.setLayout(new BorderLayout());

        // Einstellungen des mxGraph
        mxgraph = new mxGraph();
        mxgraph.setCellsEditable(false);
        mxgraph.setAllowDanglingEdges(false);
        mxgraph.setConnectableEdges(false);
        mxgraph.setAllowLoops(false);
        mxgraph.setCellsDeletable(false);
        mxgraph.setCellsCloneable(false);
        mxgraph.setCellsDisconnectable(false);
        mxgraph.setCellsResizable(false);
        mxgraph.setAutoSizeCells(false);
        mxgraph.setDropEnabled(false);
        mxgraph.setSplitEnabled(false);
        mxgraph.setCellsBendable(false);
        mxgraph.setDisconnectOnMove(false);
        mxgraph.setMultigraph(false);

        // Kontrolle, ob Kante zulässig ist
        checks[0] = new EdgeCheck();
        mxgraph.setMultiplicities(checks);

        gc = new mxGraphComponent(mxgraph); 

        // Mouse- und KeyEvent vom GraphControlFenster
        gc.getGraphControl().addMouseListener(new MouseAdapter()
            {
                public void mouseReleased(MouseEvent e) {graphControlMouseReleased(e);}

                public void mousePressed(MouseEvent e)  {graphControlMousePressed(e);}  
            });

        gc.addKeyListener(new KeyAdapter()
            {
                public void keyPressed(KeyEvent e) {graphControlKeyPressed(e);}
            });

        gc.getGraphControl().addMouseMotionListener(new MouseAdapter()
            {
                public void mouseDragged(MouseEvent e){
                    mxParallelEdgeLayout layout = new mxParallelEdgeLayout(mxgraph);
                    layout.execute(mxgraph.getDefaultParent());
                }
            });

        // Panel für Graph
        jPGraph.add(gc);
        jPGraph.setPreferredSize(new Dimension(500,340));
        jPGraph.setOpaque(true);
        jPGraph.setVisible(true);
        jPGraph.setBackground(new Color(0xC0C0C0));
        jPGraph.setBorder(BorderFactory.createEmptyBorder(3,3,3,3));

        gc.getViewport().setOpaque(true);
        gc.setBackgroundImage(null);

        //jPGraphScrollPane.setBounds(8,23,521,353);

        cp.add(jPGraphScrollPane, BorderLayout.CENTER);

        // Textausgabe Meldungen
        JPanel bottom = new JPanel();
        bottom.setBorder(BorderFactory.createEmptyBorder(2,3,2,3));
        bottom.setLayout(new BoxLayout(bottom,  BoxLayout.  Y_AXIS));
        jTAMeldungenScrollPane.setPreferredSize(new Dimension(521, 74));
        bottom.add(jTAMeldungenScrollPane);

        jLCopyright.setText("CC3.0 BY-NC-SA: Fobi RP-KA (D. Zechnall) / ZPG IMP (T. Schaller) - V2.0");
        jLCopyright.setToolTipText("CC-Lizenz-Info: https://creativecommons.org/licenses/by/3.0/de/");
        jLCopyright.setFont(new Font("Dialog", Font.PLAIN, 10));
        jLCopyright.setOpaque(true);
        jLCopyright.setAlignmentX(Component.CENTER_ALIGNMENT);
        bottom.add(jLCopyright);

        jBCopyright2.setText("unter Verwendung der JGraph-Bibliothek (Lizenzinformation)");
        jBCopyright2.setToolTipText("für Lizenzinformationen bitte anklicken");
        jBCopyright2.setFont(new Font("Dialog", Font.PLAIN, 10));
        jBCopyright2.setOpaque(true);
        jBCopyright2.setAlignmentX(Component.CENTER_ALIGNMENT);
        jBCopyright2.addMouseListener(new MouseListener() { 
                public void mouseClicked(MouseEvent e) { 
                    jButtonLicence_ActionPerformed(e);
                }

                public void mouseExited(MouseEvent e) { }

                public void mouseEntered(MouseEvent e) { }

                public void mousePressed(MouseEvent e) { }

                public void mouseReleased(MouseEvent e) { }

            });
        bottom.add(jBCopyright2);

        cp.add(bottom,BorderLayout.PAGE_END);
        // Buttons
        JPanel buttonBox = new JPanel();
        buttonBox.setBorder(BorderFactory.createEmptyBorder(0, 20, 10, 20));
        GridBagLayout gl = new GridBagLayout();
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = new Insets(10,0,0,0);
        gbc.fill = GridBagConstraints.HORIZONTAL;
        gbc.weightx = 1.0;
        buttonBox.setLayout(gl);

        jButtonBildLaden.setText("Lade Bild");
        jButtonBildLaden.setMargin(new Insets(2, 2, 2, 2));
        jButtonBildLaden.addActionListener(new ActionListener() { 
                public void actionPerformed(ActionEvent evt) { jButtonBildLaden_ActionPerformed(evt); }
            });
        jButtonBildLaden.setToolTipText("Bild aus Datei als Hintergrund oeffnen");

        buttonBox.add(jButtonBildLaden,gbc);

        jButtonBildLoeschen.setText("Lösche Bild");
        jButtonBildLoeschen.setMargin(new Insets(2, 2, 2, 2));
        jButtonBildLoeschen.addActionListener(new ActionListener() { 
                public void actionPerformed(ActionEvent evt) { jButtonBildLoeschen_ActionPerformed(evt); }
            });
        jButtonBildLoeschen.setToolTipText("Hintergrundbild entfernen");
        gbc.gridy = 1;
        buttonBox.add(jButtonBildLoeschen,gbc);

        gbc.gridy =2;
        buttonBox.add(Box.createRigidArea(new Dimension(10,20)),gbc);

        //jButtonNeuerGraph.setBounds(544, 220, 173, 25);
        jButtonNeuerGraph.setPreferredSize(new Dimension(173, 25));
        jButtonNeuerGraph.setText("Neuer Graph");
        jButtonNeuerGraph.setMargin(new Insets(2, 2, 2, 2));
        jButtonNeuerGraph.addActionListener(new ActionListener() { 
                public void actionPerformed(ActionEvent evt) { 
                    jButtonNeuerGraph_ActionPerformed(evt);
                }
            });

        gbc.gridy = 3;
        buttonBox.add(jButtonNeuerGraph,gbc);

        jButtonDateiEinlesen.setText("Graph oeffnen");
        jButtonDateiEinlesen.setMargin(new Insets(2, 2, 2, 2));
        jButtonDateiEinlesen.addActionListener(new ActionListener() { 
                public void actionPerformed(ActionEvent evt) { 
                    jButtonDateiEinlesen_ActionPerformed(evt);
                }
            });
        jButtonDateiEinlesen.setToolTipText("Lade Graph");
        gbc.gridy = 4;
        buttonBox.add(jButtonDateiEinlesen,gbc);

        jButtonSpeichereGraph.setText("Graph speichern");
        jButtonSpeichereGraph.setMargin(new Insets(2, 2, 2, 2));
        jButtonSpeichereGraph.addActionListener(new ActionListener() { 
                public void actionPerformed(ActionEvent evt) { 
                    jButtonDateiSpeichern_ActionPerformed(evt);
                }
            });
        gbc.gridy = 5;
        buttonBox.add(jButtonSpeichereGraph,gbc);

        gbc.weighty =1.0;
        gbc.gridy = 6;
        buttonBox.add(Box.createGlue(), gbc);   

        cp.add(buttonBox, BorderLayout.LINE_END);

        //jLGraphMode.setBounds(8, 8, 521, 15);
        jLGraphMode.setText("ungerichteter, ungewichteter Graph");
        jLGraphMode.setHorizontalAlignment(SwingConstants.CENTER);
        jLGraphMode.setVerticalAlignment(SwingConstants.TOP);
        cp.add(jLGraphMode, BorderLayout.PAGE_START);

        // Menüitems für Popupmenüs definieren
        jPMKanteJMenuItemSetzeKantengewicht.addActionListener(new ActionListener() { 
                public void actionPerformed(ActionEvent evt) { 
                    jPMKanteJMenuItemSetzeKantengewicht_ActionPerformed(evt);
                }
            });

        jPMKanteJMenuItemLöscheausgewählteObjekte.addActionListener(new ActionListener() { 
                public void actionPerformed(ActionEvent evt) { 
                    jPMKanteJMenuItemLöscheausgewählteObjekte_ActionPerformed(evt);
                }
            });

        pack();
        // Ende Komponenten
        neuerGraph(false, false);

        File workingDirectory = new File(System.getProperty("user.dir"));
        jFileChooser1.setCurrentDirectory(workingDirectory);
        jFileChooser2.setCurrentDirectory(workingDirectory);
        jFileChooser3.setCurrentDirectory(workingDirectory);

        setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);

        setResizable(true);
        setVisible(true);
    } // end of public EditorGUI
    // Anfang Komponenten

    // Anfang Methoden
    public void jButtonLicence_ActionPerformed(MouseEvent evt) {

        LicenceInfo i = new LicenceInfo();
    }

    public void graphControlMouseReleased(MouseEvent e) {
        Object[] selected = mxgraph.getSelectionCells();;
        if (selected.length>0)   // cell != null
        {
            jTAMeldungen.setText("");
            for (int i=0; i<selected.length; i++) {
                if (mxgraph.getModel().isVertex(selected[i]))
                    jTAMeldungen.append("Knoten "+mxgraph.getLabel(selected[i])+" ist selektiert!\n");
                if (mxgraph.getModel().isEdge(selected[i])) {
                    jTAMeldungen.append("Kante von "+
                        ((mxCell) selected[i]).getTerminal(true).getValue()+" nach "+
                        ((mxCell) selected[i]).getTerminal(false).getValue()+
                        " ist selektiert!\n");

                }
            }
        }
        else {
            jTAMeldungen.setText("Nichts selektiert");
            if(Math.abs(pressed.getX()-e.getX())+Math.abs(pressed.getY()-e.getY())<10) {
                Object o = gc.getCellAt(e.getX(), e.getY());
                if(o == null || mxgraph.getModel().isEdge(o)) neuerKnoten(e.getX()-15, e.getY()-15);
            }
        } 
        mxParallelEdgeLayout layout = new mxParallelEdgeLayout(mxgraph);
        layout.execute(mxgraph.getDefaultParent());

        if (e.isPopupTrigger() && getAnzahlSelectedEdges()+getAnzahlSelectedVertices()>0){
            JPopupMenu menu = new JPopupMenu();
            menu.add(jPMKanteJMenuItemLöscheausgewählteObjekte);
            if (getAnzahlSelectedEdges()==1 && getAnzahlSelectedVertices()==0 && weighted) 
                menu.add(jPMKanteJMenuItemSetzeKantengewicht);
            menu.show(e.getComponent(), e.getX(), e.getY()); 
        } // end of if-else
    }

    public void graphControlMousePressed(MouseEvent e) {
        pressed = e;
        if (e.isPopupTrigger() && getAnzahlSelectedEdges()+getAnzahlSelectedVertices()>0){
            JPopupMenu menu = new JPopupMenu();
            menu.add(jPMKanteJMenuItemLöscheausgewählteObjekte);
            if (getAnzahlSelectedEdges()==1 && getAnzahlSelectedVertices()==0 && weighted ) 
                menu.add(jPMKanteJMenuItemSetzeKantengewicht);
            menu.show(e.getComponent(), e.getX(), e.getY()); 
        } // end of if-else    
    }

    public void graphControlKeyPressed(KeyEvent e) {
        if(e.getKeyCode() == KeyEvent.VK_PLUS || e.getKeyCode() == KeyEvent.VK_MINUS) {
           if(e.getKeyCode() == KeyEvent.VK_PLUS) vertexSize+=2;
           if(e.getKeyCode() == KeyEvent.VK_MINUS) vertexSize-=2;
            Object[] cells = mxgraph.getChildVertices(mxgraph.getDefaultParent());
            mxgraph.getModel().beginUpdate();
            for (Object cell : cells)
            {
              if(mxgraph.getModel().isVertex(cell)){
                mxGeometry geom = (mxGeometry) mxgraph.getModel().getGeometry(cell).clone();
                geom.setWidth(vertexSize);
                geom.setHeight(vertexSize);
                mxgraph.getModel().setGeometry(cell, geom);
            }
        }
               mxgraph.getModel().endUpdate();
                mxParallelEdgeLayout layout = new mxParallelEdgeLayout(mxgraph);
                layout.execute(mxgraph.getDefaultParent());
         return;
        }
              
           
           
        Object[] selected = mxgraph.getSelectionCells();
        if(selected.length==1) {
            Object[] cells = mxgraph.getChildVertices(mxgraph.getDefaultParent());
            Object prev = null;
            Object next = null;
            Object first = null;
            Object last = null;
            int i = 0;
            for (i=0; i< cells.length; i++)
            {
              if(mxgraph.getModel().isVertex(cells[i]) && first == null) first = cells[i];
              if(cells[i] == selected[0]) break;
              if(mxgraph.getModel().isVertex(cells[i])) prev = cells[i];
            }
            i++;
            for(;i<cells.length; i++) {
              if(mxgraph.getModel().isVertex(cells[i]) && next == null) {
                  next = cells[i];
                }
              if(mxgraph.getModel().isVertex(cells[i]) ) {
                  last = cells[i];
                }
            }
             
            if (prev == null) prev = last;
            if (next == null) next = first;
            
            
            
            mxgraph.getModel().beginUpdate();
            try {
                if(e.getKeyCode() == KeyEvent.VK_DELETE) {
                    mxgraph.getModel().remove(selected[0]);
                }
                if(mxgraph.getModel().isVertex(selected[0])) {
                    if(e.getKeyCode() == KeyEvent.VK_DOWN) {
                        mxgraph.moveCells(selected, 0, 2);
                    }
                    if(e.getKeyCode() == KeyEvent.VK_UP) {
                        mxgraph.moveCells(selected, 0, -2);
                    }
                    if(e.getKeyCode() == KeyEvent.VK_RIGHT) {
                        mxgraph.moveCells(selected, 2, 0);
                    }
                    if(e.getKeyCode() == KeyEvent.VK_LEFT) {
                        mxgraph.moveCells(selected, -2,0);
                    }
                    if(e.getKeyCode() == KeyEvent.VK_PAGE_UP && prev != null) {
                      gc.getGraph().clearSelection();  
                      gc.selectCellForEvent(prev, null) ;
                    }
                    if(e.getKeyCode() == KeyEvent.VK_PAGE_DOWN) {
                      gc.getGraph().clearSelection();  
                      gc.selectCellForEvent(next, null) ;
                    }                    
                    mxgraph.refresh();
                }
            }
            catch(Exception ex) {
                System.out.println("Fehler: "+ex);
            }
            finally {
                mxgraph.getModel().endUpdate();
                mxParallelEdgeLayout layout = new mxParallelEdgeLayout(mxgraph);
                layout.execute(mxgraph.getDefaultParent());
            }
        } else {
            jTAMeldungen.setText("Bitte genau ein Objekt auswählen");
        } // end of if-else

    }

    public void jButtonDateiSpeichern_ActionPerformed(ActionEvent evt) {

        JPanel panel = new JPanel();
        JRadioButton jRadioButtonMatrix = new JRadioButton("als Matrix");
        JRadioButton jRadioButtonListe = new JRadioButton("als Liste");
        jRadioButtonMatrix.setSelected(true);
        ButtonGroup buttonGroupRepresentation = new ButtonGroup();
        buttonGroupRepresentation.add(jRadioButtonMatrix);
        buttonGroupRepresentation.add(jRadioButtonListe);

        panel.add(new JLabel("Art der Speicherung" ));
        panel.add(jRadioButtonMatrix);
        panel.add(jRadioButtonListe);

        if(JOptionPane.showConfirmDialog(null, panel, "Graph speichern?", JOptionPane.OK_CANCEL_OPTION, JOptionPane.QUESTION_MESSAGE )==JOptionPane.OK_OPTION)
        {
            File f = jFileChooser2_saveFile();
            if (f!=null) {
                try{
                    boolean asMatrix = jRadioButtonMatrix.isSelected();
                    String text = "";
                    if (asMatrix) text += "Der Graph liegt hier in Form einer Adjazenzmatrix vor.";
                    else text += "Der Graph liegt hier in Form einer Adjazenzliste vor.";
                    text += " Jede Zeile steht fuer einen Knoten, durch Komma getrennt steht der adjazente Knoten mit dem zugehoerigen Kantengewicht.";
                    if(writeCSV(f.getAbsolutePath(), text, asMatrix)) {
                        jTAMeldungen.setText("CSV-Datei wurde erfolgreich erzeugt!");
                        this.setTitle(guiTitle+" - "+f.getName());
                    }
                    else {
                        jTAMeldungen.setText("Fehler beim Erzeugen der CSV-Datei!");
                    } // end of if-else
                }
                catch (SecurityException e) {}
            }
        }
    } // end of jButtonSpeichereGraph_ActionPerformed

    public void jButtonDateiEinlesen_ActionPerformed(ActionEvent evt) {
        File f = jFileChooser1_openFile();
        if (f!=null) {
            try{
                if(csvDateiEinlesen(f.getAbsolutePath())) {
                    this.setTitle(guiTitle+" - "+ f.getName());
                    jTAMeldungen.setText("Datei erfolgreich eingelesen!");

                    this.jPGraph.repaint();
                }
                else jTAMeldungen.setText("Fehler beim Einlesen der Datei!");
            }
            catch (SecurityException e) {}
        }
    } // end of jButtonDateiEinlesen_ActionPerformed

    public void jButtonBildLaden_ActionPerformed(ActionEvent evt) {
        String workingDir = System.getProperty("user.dir");
        File f = jFileChooser3_openFile();
        if (f!=null) {
            try{
                Files.copy( f.toPath(), Paths.get(workingDir,"images",f.getName()), REPLACE_EXISTING);
                bildDatei = f.getName();
                ImageIcon img = new ImageIcon("images/"+bildDatei);
                gc.setBackgroundImage(img);
                jPGraph.setPreferredSize(new Dimension(img.getIconWidth(),img.getIconHeight()));
                jPGraphScrollPane.revalidate();
                this.jPGraphScrollPane.repaint();
            }
            catch (Exception e) {}
        }
    } // end of jButtonBildLaden_ActionPerformed

    public void jButtonBildLoeschen_ActionPerformed(ActionEvent evt) {
        bildDatei = "";
        gc.setBackgroundImage(null);
        this.jPGraph.repaint();
    } // end of jButtonBildLaden_ActionPerformed

    public void jButtonNeuerGraph_ActionPerformed(ActionEvent evt) {
        JPanel panel = new JPanel();

        JRadioButton button1 = new JRadioButton("gerichtet");
        JRadioButton button2 = new JRadioButton("ungerichtet");
        button2.setSelected(true);
        ButtonGroup buttonGroupDirected = new ButtonGroup();
        buttonGroupDirected.add(button1);
        buttonGroupDirected.add(button2);

        JRadioButton button3 = new JRadioButton("gewichtet");
        JRadioButton button4 = new JRadioButton("ungewichtet");
        button4.setSelected(true);
        ButtonGroup buttonGroupWeighted = new ButtonGroup();
        buttonGroupWeighted.add(button3);
        buttonGroupWeighted.add(button4);

        panel.add(button1);
        panel.add(button2);
        panel.add(new JLabel(" - " ));
        panel.add(button3);
        panel.add(button4);

        if(JOptionPane.showConfirmDialog(null, panel, "Neuen Graph erstellen?", JOptionPane.OK_CANCEL_OPTION, JOptionPane.QUESTION_MESSAGE )==JOptionPane.OK_OPTION)
        {
            neuerGraph(button1.isSelected(), button3.isSelected());
            this.setTitle(guiTitle+" - "+"ungespeichertes Dokument");
            jTAMeldungen.setText("Graph erfolgreich initialisiert!");
        }
    } // end of jButtonNeuerGraph_ActionPerformed

    public void jPMKanteJMenuItemSetzeKantengewicht_ActionPerformed(ActionEvent evt) {
        Object[] selected = mxgraph.getSelectionCells();

        String gewichtS = JOptionPane.showInputDialog(this, "Geben Sie bitte das Kantengewicht ein");

        mxgraph.getModel().beginUpdate();
        try {
            Double gewicht = Double.parseDouble(gewichtS);
            for (int i=0; i<selected.length ; i++ ) {
                if (mxgraph.getModel().isEdge(selected[i])) {
                    mxgraph.getModel().setValue(selected[i], gewicht);
                    jTAMeldungen.append("Kante mit Gewicht "+mxgraph.getLabel(selected[i])+" ist selektiert!\n");
                }
            } // end of for
        }

        catch(Exception e) {
            System.out.println("Fehler: "+e);
        }
        finally {
            mxgraph.getModel().endUpdate();
            mxParallelEdgeLayout layout = new mxParallelEdgeLayout(mxgraph);
            layout.execute(mxgraph.getDefaultParent());
        }

    } // end of jPMKanteJMenuItemSetzeKantengewicht_ActionPerformed

    public void jPMKanteJMenuItemLöscheausgewählteObjekte_ActionPerformed(ActionEvent evt) {
        Object[] selected = mxgraph.getSelectionCells();

        mxgraph.getModel().beginUpdate();
        try {
            for (int i=0; i<selected.length ; i++ ) {
                mxgraph.getModel().remove(selected[i]);
            } // end of for
        }                                              
        catch(Exception e) {
            System.out.println("Fehler: "+e);
        }
        finally {
            mxgraph.getModel().endUpdate();
            mxParallelEdgeLayout layout = new mxParallelEdgeLayout(mxgraph);
            layout.execute(mxgraph.getDefaultParent());
        }

    } // end of jPMKanteJMenuItemLöscheausgewählteObjekte_ActionPerformed

    public File jFileChooser1_openFile() {
        if (jFileChooser1.showOpenDialog(this) == JFileChooser.APPROVE_OPTION) {
            return jFileChooser1.getSelectedFile();
        } else {
            return null;
        }
    }

    public File jFileChooser2_saveFile() {
        if (jFileChooser2.showSaveDialog(this) == JFileChooser.APPROVE_OPTION) {
            return jFileChooser2.getSelectedFile();
        } else {
            return null;
        }
    }

    public File jFileChooser3_openFile() {
        if (jFileChooser3.showOpenDialog(this) == JFileChooser.APPROVE_OPTION) {
            return jFileChooser3.getSelectedFile();
        } else {
            return null;
        }
    }

    public void setVertexStyle() {
        Map<String, Object> vertexStyle = new HashMap<String, Object>();

        vertexStyle.put(mxConstants.STYLE_SHAPE,   mxConstants.SHAPE_ELLIPSE);
        vertexStyle.put(mxConstants.STYLE_PERIMETER, mxConstants.PERIMETER_ELLIPSE);
        vertexStyle.put(mxConstants.STYLE_STROKECOLOR, "#000080");
        vertexStyle.put(mxConstants.STYLE_STROKEWIDTH, "2");
        vertexStyle.put(mxConstants.STYLE_FONTCOLOR, "#000080");
        vertexStyle.put(mxConstants.STYLE_FILLCOLOR, "lightgrey");

        mxStylesheet styleSheet = mxgraph.getStylesheet();
        styleSheet.setDefaultVertexStyle(vertexStyle);
    }

    public void setEdgeStyle(boolean directed) {
        Map<String, Object> edgeStyle = new HashMap<String, Object>();

        edgeStyle.put(mxConstants.STYLE_SHAPE,    mxConstants.SHAPE_CONNECTOR);
        edgeStyle.put(mxConstants.STYLE_STROKECOLOR, "#000080");
        edgeStyle.put(mxConstants.STYLE_STROKEWIDTH, "2");
        edgeStyle.put(mxConstants.STYLE_FONTCOLOR, "#000000");
        edgeStyle.put(mxConstants.STYLE_LABEL_BACKGROUNDCOLOR, "#ffffff");

        if (!directed) {
            edgeStyle.put(mxConstants.STYLE_ENDARROW, mxConstants.NONE);
            ((EdgeCheck) checks[0]).setGerichtet(false);
        } else {
            edgeStyle.put(mxConstants.STYLE_ENDARROW, mxConstants.ARROW_CLASSIC);
            ((EdgeCheck) checks[0]).setGerichtet(true);
        } // end of if-else

        mxStylesheet styleSheet = mxgraph.getStylesheet();
        styleSheet.setDefaultEdgeStyle(edgeStyle);
    }

    public int getAnzahlSelectedEdges() {
        Object[] selected = mxgraph.getSelectionCells();;
        int anz = 0;
        for (int i=0; i<selected.length; i++) {
            if (mxgraph.getModel().isEdge(selected[i])) {
                anz++;
            }
        }
        return anz;
    }

    public int getAnzahlSelectedVertices() {
        Object[] selected = mxgraph.getSelectionCells();;
        int anz = 0;
        for (int i=0; i<selected.length; i++) {
            if (mxgraph.getModel().isVertex(selected[i])) {
                anz++;
            }
        }
        return anz;
    }

    public void neuerKnoten(int x, int y) {
        boolean geschafft = false;
        mxgraph.getModel().beginUpdate();
        try {
            // Knoten erstellen
            Object k2;
            nrKnoten++;
            if(showVertexText)
                k2 = mxgraph.insertVertex(mxgraph.getDefaultParent(), null, "K"+this.nrKnoten, x, y, vertexSize, vertexSize, "");
            else
                k2 = mxgraph.insertVertex(mxgraph.getDefaultParent(), null, null, x, y, vertexSize, vertexSize, "");
        }

        catch(Exception e) {
            geschafft = false;
        }
        finally {
            mxgraph.getModel().endUpdate();
            mxParallelEdgeLayout layout = new mxParallelEdgeLayout(mxgraph);
            layout.execute(mxgraph.getDefaultParent());
        }

        jPGraph.repaint();
    }

    public void neuerGraph(boolean directed, boolean weighted) {
        neuerGraph(directed, weighted, 30, false, true);
    }

    public void neuerGraph(boolean directed, boolean weighted, int vertexSize, boolean showVertexValue, boolean showVertexText){
        this.directed = directed;
        this.weighted = weighted;

        mxgraph.getModel().beginUpdate();
        try {
            // alles löschen
            mxgraph.selectAll();
            Object[] selected = mxgraph.getSelectionCells();
            for(Object o : selected) {
                mxgraph.getModel().remove(o);
            }

            // Wieder bei 1 beginnen mit Nummerierung
            nrKnoten=0;
            setVertexStyle();
            setEdgeStyle(directed);

            // Anzeige anpassen
            String s = "";
            if (directed) {
                s += "gerichteter, ";
            } else {
                s += "ungerichteter, ";
            } // end of if-else
            if (weighted) {
                s+= "gewichteter Graph";
                showEdgeWeights = true;
            } else {
                s+="ungewichteter Graph";
                showEdgeWeights = false;
            } // end of if-else
            jLGraphMode.setText(s);
            this.vertexSize = vertexSize;
            this.showVertexValue = showVertexValue;
            this.showVertexText = showVertexText;
        }
        catch(Exception e) {
            System.out.println("Fehler: "+e);
        }
        finally {
            mxgraph.getModel().endUpdate();
            mxParallelEdgeLayout layout = new mxParallelEdgeLayout(mxgraph);
            layout.execute(mxgraph.getDefaultParent());
        }
    }

    /**
     * Eine CSV-Datei als Matrix wird eingelesen (geoeffnet).
     *
     * @param   String[][] matrix   die Matrix
     * @return  boolean geklappt (true/false)
     */
    public boolean erzeugeGraphAusMatrix(String[][] matrix) {
        boolean geschafft = false;
        try {
            int knotenAnzahl = 0;
            geschafft = true;

            mxgraph.getModel().beginUpdate();

            // Knoten erstellen
            for(int i = 0; i < matrix.length; i++) {
                String[] row = matrix[i];
                if(row[0].charAt(0) != '#'){
                    knotenAnzahl++;
                    neuerKnoten(Integer.valueOf(row[0]), Integer.valueOf(row[1]));
                }
            }


            // Kanten erstellen
            Object[] knoten = mxgraph.getChildVertices(mxgraph.getDefaultParent());

            int k=0;
            for(int i = 0; i < matrix.length; i++) {
                String[] row = matrix[i];
                if(row[0].isEmpty() || row[0].charAt(0) != '#') { //Ueberpruefung auf Kommentare #
                    for(int j = 2; j < row.length; j++) {
                        if(!(row[j].equals("-"))) {
                            Object[] alteKanten = mxGraphModel.getEdgesBetween(mxgraph.getModel(),knoten[k], knoten[j-2], directed);
                            if(alteKanten == null || alteKanten.length == 0) {
                                Object o;
                                if(Double.valueOf(row[j]) == Double.NaN || !this.weighted) 
                                    o = mxgraph.insertEdge(mxgraph.getDefaultParent(), null, null, knoten[k], knoten[j - 2], null);
                                else 
                                    o = mxgraph.insertEdge(mxgraph.getDefaultParent(), null, Double.valueOf(row[j]), knoten[k], knoten[j - 2], null);
                            }
                        }
                    }
                    k++;
                }
            }
        }
        catch(Exception e) {
            geschafft = false;
        }
        finally {
            mxgraph.getModel().endUpdate();
            mxParallelEdgeLayout layout = new mxParallelEdgeLayout(mxgraph);
            layout.execute(mxgraph.getDefaultParent());
        }
        return geschafft;
    }

    /**
     * Eine CSV-Datei als Liste wird eingelesen (geoeffnet).
     *
     * @param   String[][] matrix   die Liste
     * @return  boolean geklappt (true/false)
     */
    public boolean erzeugeGraphAusListe(String[][] matrix) {
        boolean geschafft = false;
        try {
            int knotenAnzahl = 0;
            geschafft = true;

            mxgraph.getModel().beginUpdate();

                // Knoten erstellen
                for(int i = 0; i < matrix.length; i++) {
                    String[] row = matrix[i];
                    if(row[0].charAt(0) != '#') { // Ueberpruefung auf Kommentare #
                        if(row[0].charAt(0) != '~')
                            neuerKnoten(Integer.valueOf(row[0]), Integer.valueOf(row[1]));
                    }
                }

                // Kanten einlesen
                Object[] knoten = mxgraph.getChildVertices(mxgraph.getDefaultParent());

                int k = 0;
                for(int i = 0; i < matrix.length; i++) {
                    String[] row = matrix[i];
                    if(row[0].isEmpty() || row[0].charAt(0) != '#') { //Ueberpruefung auf Kommentare #
                        for(int j = 2; j < row.length; j=j+2) {
                            Object[] alteKanten = mxGraphModel.getEdgesBetween(mxgraph.getModel(),knoten[k], knoten[Integer.valueOf(row[j])-1], directed);
                            if(alteKanten == null || alteKanten.length == 0) {
                                Object o;
                                if(Double.valueOf(row[j + 1]) == Double.NaN || !this.weighted) 
                                    o = mxgraph.insertEdge(mxgraph.getDefaultParent(), null, null, knoten[k], knoten[Integer.valueOf(row[j])-1], null);
                                else 
                                    o = mxgraph.insertEdge(mxgraph.getDefaultParent(), null, Double.valueOf(row[j+1]), knoten[k], knoten[Integer.valueOf(row[j])-1], null);
                            }
                        }
                        k++;
                    }
                }
        }
        catch(Exception e) {
            geschafft = false;
        }
        finally {
            mxgraph.getModel().endUpdate();
            mxParallelEdgeLayout layout = new mxParallelEdgeLayout(mxgraph);
            layout.execute(mxgraph.getDefaultParent());
        }
        return geschafft;
    }

    
    /**
     * Eine CSV-Datei als Liste wird eingelesen (geoeffnet). Der DateiName wird angegeben.
     *
     * @param   String dateiName   Der DateiName
     * @return  boolean geklappt (true/false)
     */ 
    public boolean csvDateiEinlesen(String dateiName) {
        boolean geschafft = false;
        try {
            CSVParser csvParser = new CSVParser(Files.newInputStream(Paths.get(dateiName)));
            if(csvParser != null) {
                String[][] matrix = csvParser.getAllValues();
                csvParser.close();

                farbenKnoten = stdFarbenKnoten;
                farbenKanten = stdFarbenKanten;
                vertexSize = 30;
                directed = false;
                weighted = false;
                showEdgeWeights = false;
                showVertexText = true;
                showVertexValue = false;
                bildDatei = "";

                geschafft = true;
                int i;
                for(i = 0; i < matrix.length; i++) {
                    String[] row = matrix[i];

                    if(row[0].equals("directed")) {
                        directed = (row[1].charAt(0) == '1');
                    }
                    if(row[0].equals("weighted")) {
                        weighted = (row[1].charAt(0) == '1');
                    }
                    if(row[0].equals("showWeights")) {
                        showEdgeWeights = (row[1].charAt(0) == '1');
                    }
                    if(row[0].equals("vertexSize")) {
                        vertexSize = Integer.parseInt(row[1]);
                    }
                    if(row[0].equals("vertexStyle")) {
                        showVertexText = (row[1].charAt(0) == '1');
                        showVertexValue = (row[1].charAt(0) == '2');
                    }
                    if(row[0].equals("image")) {
                        if(row[1].charAt(0) == '0')
                            bildDatei = "";
                        else
                            bildDatei = row[1];
                    }
                    if(row[0].equals("vertexColor")) {
                        farbenKnoten = Arrays.copyOfRange(row, 1, row.length);  
                    }
                    if(row[0].equals("edgeColor")) {
                        farbenKanten = Arrays.copyOfRange(row, 1, row.length);  
                    }
                    if(row[0].equals("matrix") || row[0].equals("list")) {
                        break;
                    }

                }

                neuerGraph(directed, weighted, vertexSize, showVertexValue, showVertexText);
                
                if(matrix[i][0].equals("matrix")) {
                    geschafft = erzeugeGraphAusMatrix(Arrays.copyOfRange(matrix,i+1,matrix.length));
                } else {
                    geschafft = erzeugeGraphAusListe(Arrays.copyOfRange(matrix,i+1,matrix.length));
                }


                

                // Alle Komponenten entfernen und neu aufbauen
                   if(bildDatei.equals("")){
                      gc.setBackgroundImage(null);
                   } else {
                      gc.setBackgroundImage(new ImageIcon("images/"+bildDatei));
                   }

                repaint();
            } 
        }
        catch(Exception e) {
            geschafft = false;
        }
        return geschafft;
 
        
        



   
    }

    /**
     * Schreibt die CSV-Datei mit einem Header. Je nach asMatrix-Wert wird der Graph als Adjazenzliste oder als -Matrix gespeichert.
     * 
     * @param   String header   Der Header
     * @param   boolean asMatrix   Matrix (true), Liste (false)
     * @return  boolean geklappt (true/false)
     */
    public boolean writeCSV(String header, boolean asMatrix) {
        return writeCSV("graphs\\savedGraph.csv", header, asMatrix);
    }

    /**
     * Schreibt die CSV-Datei mit einem Header. Je nach asMatrix-Wert wird der Graph als Adjazenzliste oder als -Matrix gespeichert.
     * Der Dateiname wird hier mit angegeben.
     * 
     * @param   String dateiName   Der DateiName
     * @param   String header   Der Header
     * @param   boolean asMatrix   Matrix (true), Liste (false)
     * @return  boolean geklappt (true/false)
     */
    public boolean writeCSV(String dateiName, String header, boolean asMatrix) {
        return writeGraph(dateiName, toCSVString(asMatrix), "Der Graph ist durch kommagetrennte Werte kodiert. "+header, "Ende", asMatrix);
    }

    /**
     * Schreibt die CSV-Datei mit einem Header, Footer und Text in die Datei mit dem Namen DateiName.
     * Je nach asMatrix-Wert wird der Graph als Adjazenzliste oder als -Matrix gespeichert.
     * 
     * @param   String dateiName   Der DateiName
     * @param   String text   Der Text
     * @param   String header   Der Header
     * @param   boolean asMatrix   Matrix (true), Liste (false)
     * @return  boolean geklappt (true/false)
     */
    public boolean writeGraph(String dateiName, String text, String header, String footer, boolean asMatrix) {
        try {
            String gesamtText = "# "+header + "\n";
            gesamtText += "# gerichtet 1, ungerichtet 0\n";
            if(directed)
                gesamtText += "directed,1\n";
            else
                gesamtText += "directed,0\n";
            gesamtText += "# gewichtet 1, ungewichtet 0\n";
            if(weighted)
                gesamtText += "weighted,1\n";
            else
                gesamtText += "weighted,0\n";
            gesamtText += "# Gewichte anzeigen 1, Gewichte nicht anzeigen 0\n";
            if(showEdgeWeights)
                gesamtText += "showWeights,1\n";
            else
                gesamtText += "showWeights,0\n";
            if(vertexSize != 30) {
               gesamtText += "# Größe der Knoten\n";
               gesamtText += "vertexSize,"+vertexSize+"\n";
            }
            
            gesamtText += "# Knoten leer 0, Knotenname anzeigen 1, Wert des Knoten anzeigen 2\n";
            if(!showVertexText && !showVertexValue) gesamtText += "vertexStyle,0\n";
            else {
                if(showVertexText) gesamtText += "vertexStyle,1\n";
                else
                    gesamtText += "vertexStyle,2\n";
            }
            if(farbenKanten!=stdFarbenKanten) {
                gesamtText += "# Kantenfarben: Farbbeschreibung (z.B. red) oder RGB-Hexcode (z.B. #FF0000) oder invisible\n";
                gesamtText += "# Reihenfolge: normale Kanten, markierte Kanten, gelöschte Kanten\n";
                gesamtText += "edgeColor";
                for(String f : farbenKanten) gesamtText+=","+f;
                gesamtText += "\n";
            }
            if(farbenKnoten!=stdFarbenKnoten) {
                gesamtText += "# Knotenfarben: Farbbeschreibung (z.B. red) oder RGB-Hexcode (z.B. #FF0000) oder invisible\n";
                gesamtText += "# Reihenfolge: markiert (+1), besucht(+2), beendet(+4), z.B. markiert+beender=>6.Farbe \n";
                gesamtText += "vertexColor";
                for(String f : farbenKnoten) gesamtText+=","+f;
                gesamtText += "\n";
            }
            gesamtText += "# Bild im Hintergrund (bitte im \"images\"-Ordner ablegen) --> Dateiname angeben. Fall kein Bild bitte 0 schreiben!\n";
            if(bildDatei.equals(""))
                gesamtText += "image,0\n";
            else
                gesamtText += "image,"+bildDatei + "\n";
            gesamtText += "#\n# Graph:\n";
            if(asMatrix) gesamtText +="matrix\n";
            else gesamtText += "list\n";
            gesamtText += text + "# "+footer + "\n";
            String name = dateiName.substring(dateiName.lastIndexOf("\\")+1);
            if(name.contains(".")) dateiName = dateiName.substring(0, dateiName.lastIndexOf("."));
            FileUtils.writeStringToFile(new File(dateiName + ".csv"), gesamtText);
            return true;
        } catch(Exception e) {
            return false;
        }
    }

    /**
     * Die Methode erstellt eine CSV-Ausgabe des Graphen entweder als Adjazenzliste oder als Adjazenzmatrix.
     *
     * @param   boolean asMatrix    true, falls die CSV-Ausgabe eine AdjazenzMatrix sein soll, sonst false
     * @return  String  CSV-Ausgabe
     */
    public String toCSVString(boolean asMatrix) {
        String s="";
        Object[] knoten = mxgraph.getChildVertices(mxgraph.getDefaultParent());
        if(knoten == null || knoten.length==0) return s;
        for (int a=0; a<knoten.length; a++) {
            s+=((int)((mxCell)knoten[a]).getGeometry().getX())+","+((int)((mxCell)knoten[a]).getGeometry().getY())  ;
            for (int z=0; z<knoten.length; z++) {
                Object[] kante = mxGraphModel.getEdgesBetween(mxgraph.getModel(),knoten[a], knoten[z], directed); 
                if(kante == null || kante.length==0) {
                    if(asMatrix) {
                        s += ",-";
                    } 
                } else {
                    Double d;
                    try{
                        d = (Double) (mxgraph.getModel().getValue(kante[0]));
                        if(d==null) d = 0.0;
                    } catch (Exception e) {
                        d = 0.0;
                    }
                    if(asMatrix) {
                        s += ","+d;            
                    } else {
                        s += ","+(z+1)+","+d;
                    } // end of if-else

                } // end of if-else

            } // end of for
            s+="\n";
        } // end of for

        return s;
    }

    // Ende Methoden
    public static void main(String[] arg) {
        new EditorGUI(); 

    }

} // end of class GraphGUI
